# rssi_power_database.py
# Versão: 1.0
# Data: 15/08/2025
# Descrição: Módulo de Gerenciamento de Banco de Dados para o RSSIPowerModule.
#            Armazena resultados de testes de varredura de potência.

import json
import os
from datetime import datetime
from typing import List, Dict, Optional

class RSSIPowerDatabase:
    def __init__(self, db_file: str = None):
        """
        Inicializa o banco de dados de testes RSSI vs Potência.
        
        Args:
            db_file: Caminho para o arquivo JSON do banco de dados.
        """
        if db_file is None:
            current_dir = os.path.dirname(os.path.abspath(__file__))
            self.db_file = os.path.join(current_dir, "rssi_power_db.json")
        else:
            self.db_file = db_file
        
        self.data = self._load_data()
    
    def _load_data(self) -> Dict:
        """Carrega os dados do arquivo JSON."""
        try:
            if os.path.exists(self.db_file):
                with open(self.db_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    if isinstance(data, dict) and "test_history" in data:
                        return data
            # Se não existir ou estiver corrompido, cria uma nova estrutura
            return self._create_initial_structure()
        except (json.JSONDecodeError, IOError):
            return self._create_initial_structure()

    def _create_initial_structure(self) -> Dict:
        """Cria a estrutura de dados inicial para o banco de dados."""
        return {
            "test_history": [],
            "metadata": {
                "created": datetime.now().strftime("%d-%m-%Y %H:%M"),
                "version": "1.0"
            }
        }

    def _save_data(self) -> bool:
        """Salva os dados no arquivo JSON."""
        try:
            self.data["metadata"]["last_modified"] = datetime.now().strftime("%d-%m-%Y %H:%M")
            with open(self.db_file, 'w', encoding='utf-8') as f:
                json.dump(self.data, f, indent=2, ensure_ascii=False)
            return True
        except Exception as e:
            print(f"Erro ao salvar banco de dados RSSI vs Potência: {e}")
            return False
    
    def get_test_history(self) -> List[Dict]:
        """Retorna o histórico de testes."""
        self.data = self._load_data()
        return self.data.get("test_history", [])
    
    def add_test_to_history(self, test_data: Dict) -> bool:
        """
        Adiciona um teste ao histórico preservando testes anteriores
        
        Args:
            test_data: Dados do teste
            
        Returns:
            True se adicionou com sucesso, False caso contrário
        """
        if "test_history" not in self.data:
            self.data["test_history"] = []
        
        # Gera ID único sequencial para o novo teste
        existing_ids = [test.get("id", 0) for test in self.data["test_history"]]
        test_id = max(existing_ids, default=0) + 1
        test_data["id"] = test_id
        
        # Adiciona timestamp se não existir
        if "timestamp" not in test_data:
            test_data["timestamp"] = datetime.now().strftime("%d-%m-%Y %H:%M")
        
        # Adiciona o novo teste ao histórico (preservando os anteriores)
        self.data["test_history"].append(test_data)
        return self._save_data()
    
    def update_test_in_history(self, test_id: int, test_data: Dict) -> bool:
        """
        Atualiza um teste específico no histórico
        
        Args:
            test_id: ID do teste
            test_data: Novos dados do teste
            
        Returns:
            True se atualizou com sucesso, False caso contrário
        """
        for i, test in enumerate(self.data["test_history"]):
            if test.get("id") == test_id:
                test_data["id"] = test_id
                # Preserva timestamp original se não fornecido
                if "timestamp" not in test_data:
                    test_data["timestamp"] = test.get("timestamp", "")
                self.data["test_history"][i] = test_data
                return self._save_data()
        return False
    
    def delete_test_from_history(self, test_id: int) -> bool:
        """
        Remove um teste específico do histórico
        
        Args:
            test_id: ID do teste
            
        Returns:
            True se removeu com sucesso, False caso contrário
        """
        try:
            # Remove apenas o teste específico do histórico
            original_count = len(self.data["test_history"])
            self.data["test_history"] = [test for test in self.data["test_history"] if test.get("id") != test_id]
            
            if len(self.data["test_history"]) < original_count:
                print(f"🗑️ Teste ID {test_id} removido do histórico RSSI x Potência")
                return self._save_data()
            else:
                print(f"⚠️ Teste ID {test_id} não encontrado no histórico")
                return False
        except Exception as e:
            print(f"❌ Erro ao remover teste do histórico RSSI x Potência: {e}")
            return False
    
    def get_test_by_id(self, test_id: int) -> Optional[Dict]:
        """
        Busca um teste específico por ID
        
        Args:
            test_id: ID do teste
            
        Returns:
            Dicionário com os dados do teste ou None se não encontrado
        """
        test_history = self.data.get("test_history", [])
        for test in test_history:
            if test.get("id") == test_id:
                return test
        return None

    def get_statistics(self) -> Dict:
        """Retorna estatísticas simples sobre os testes."""
        tests = self.data.get("test_history", [])
        total_tests = len(tests)
        unique_tags = set(t.get('tag_epc') for t in tests if t.get('tag_epc'))
        
        return {
            "total_tests": total_tests,
            "unique_tags_tested": len(unique_tags)
        }